
# This file is part of fbget: Python wrappers for the FireBrowse RESTful api
# Autogenerated on 2017_08_17 16:45:01 EDT

__version__ = '0.1.9'

# Copyright (c) 2015-2016, Broad Institute, Inc. {{{
# All rights reserved.
#
# This file is part of fbget: Python wrappers for the FireBrowse RESTful api
#
# FBGET is distributed under the following BSD-style license:
#
# Copyright (c) 2015-2016, Broad Institute, Inc.  All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1. Redistributions of source code must retain the above copyright notice, this
#    list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# 3. Neither the name of the Broad Institute, Inc. nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED "AS IS."  BROAD MAKES NO EXPRESS OR IMPLIED
# REPRESENTATIONS OR WARRANTIES OF ANY KIND REGARDING THE SOFTWARE AND
# COPYRIGHT, kINCLUDING, BUT NOT LIMITED TO, WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE, CONFORMITY WITH ANY DOCUMENTATION,
# NONINFRINGEMENT, OR THE ABSENCE OF LATENT OR OTHER DEFECTS, WHETHER OR NOT
# DISCOVERABLE. IN NO EVENT SHALL BROAD, THE COPYRIGHT HOLDERS, OR CONTRIBUTORS
# BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF,
# HAVE REASON TO KNOW, OR IN FACT SHALL KNOW OF THE POSSIBILITY OF SUCH DAMAGE.
#
# If, by operation of law or otherwise, any of the aforementioned warranty
# disclaimers are determined inapplicable, your sole remedy, regardless of the
# form of action, including, but not limited to, negligence and strict
# liability, shall be replacement of the software with an updated version if
# one exists.
#
# In addition, FBGET is distributed, in part, under and subject to the
# provisions of licenses for:
#
#   Python requests library
#   (http://docs.python-requests.org/en/latest/user/intro),
#   Copyright (c) 2015 Kenneth Reitz (all rights reserved); and

#   Python 2.7.9
#   (https://docs.python.org/3/license.html),
#   Copyright (c) 2001-2015 Python Software Foundation (all rights reserved).
#
# Development of FBGET has been funded in whole or in part with federal funds
# from the National Institutes of Health, Department of Health and Human
# Services, under Grant No. U24 CA143845-03.
#
# }}}
 
from fbmap import *
from fbcore import *

@autohelp(False)
def param_values(name):
    '''
    For the given named API parameter, indicate the values it may
    hold.  Most parameters are case insensitive (e.g. gene, cohort,
    TCGA barcode, tool), but some are not (e.g. column names in a
    MAF).  A parameter which admits multiple values (e.g. the gene of
    interest in mRNASeq expression samples) may be specified as a

        string     with values delimited by commas or whitespace
    OR
        sequence (e.g. list, tuple, set) of strings

    The api defines 'optionality' as the degree of 'required-ness' that
    a parameter possesses within a given signature: it may be one of

        required:   (mandatory) every required param must be specified
                    for each call, in positional form only.  There are
                    only a few of these exposed by the API.
    semiRequired:   at least one of set of semi-Required parameters, if
                    any, must be specified for each call; they may be
                    given in either positional or keyword form;  the
                    documentation for both the RESTful api and the Python
                    client bindings lists the semiRequired paramters.
        optional:   may always be omitted from any call

    Recall that in Python positional parameters MUST ALWAYS be specified
    first in the argument list, in exactly the order shown in the
    documentation.  Keyword arguments may appear in any order, as long as
    they ALL appear AFTER any of the required args.  Finally, note that
    "self" and "kwargs" are transparently inserted by Python at runtime,
    so you SHOULD NOT specify them explicitly in your calls.
    '''

    param = SupportedParameters.get(name, None)
    if not param:
        print "No such parameter '%s' is supported by the FireBrowse api" % name
        return

    values = ", ".join(param["values"])
    values = "The %s parameter may take the values:  %s" % (name, values)
    print values
    d = param["default"]

    # When list of values is long, e.g. tier1_cde_name, add whitespace
    if len(values) > 80:
        print

    if d:
        print "When not specified in an api call, its default value will be: ",d
    else:
        print "There is no default value for this parameter."

@autohelp(False)
def param_multivalued(name):
    ''' Does given named param permit >1 value to be specified?'''
    return SupportedParameters[name]["multiValued"]

def param_names():
    ''' List the names of all parameters offered by the API '''
    return (', '.join(SupportedParameters.keys())) + '\n'

def parse_args(args, argspec):
    # Partition the set of args passed to the calling Python function into
    #   restParams:   params array sent to RESTful api call on server
    #   clientParams: params dict passed to client func that issues RESTful call

    restParams = []
    clientParams = {}

    # Merge keyword (optional) args, if any, and positional (required) args
    kwargs = args.pop("kwargs", None)
    if kwargs:
        args.update(kwargs)

    # Respect defaults of containing object, for parameters that were omitted
    if "format" in argspec.queryParams and "format" not in args:
        args["format"] = argspec.caller.default_format

    if "page" in argspec.queryParams and "page" not in args:
        args["page"] = argspec.caller.default_page

    if "page_size" in argspec.queryParams and "page_size" not in args:
        args["page_size"] = get_page_size()

    # Elide anything else that should NOT be mistaken for a REST or client param
    args.pop("self", None)

    def normalize_multivalue(v, delim=','):
        # Accommodate messy input: translate everything to string and
        # tokenize by comma and/or whitespace delimiters (and strip
        # both kinds of delimiters from both ends of value string)
        v = str(v).strip(" \t,")
        return delim.join(re.split(r'[ \t,]+', v))

    for param, val in args.iteritems():

        # Ignore parameters specified with value=None
        if val == None:
            continue

        # Multiple values can be given as CSV (in a single string) or iterables
        if hasattr(val, '__iter__'):
            val = ",".join(map(str, val))
        else:
            val = str(val)
        if param in argspec.queryParams:
            if param == "format":
                clientParams["codec"] = val
                if val.lower() == CODEC_DJSON:
                    val = CODEC_JSON
            elif param == "page" and int(val) < 0:
                clientParams["pages"] = PAGES_ALL
                continue
            elif param_multivalued(param):
                val = normalize_multivalue(val)

            param = ParamAliases.get(param, param)
            restParams.append("%s=%s" % (param, val))
        elif param in argspec.pathParams:
            # Path params may be multivalued, too, so delimit them if so
            args[param] = normalize_multivalue(val, delim='%2C')
            continue
        else:
            raise TypeError("Unsupported method arg: %s" % param)

    restParams = "&".join(restParams)
    if restParams:
          restParams = "?" + restParams
    return restParams, clientParams

class Samples (object):
    ''' Fine grained retrieval of sample-level data '''

    def __init__(self, format=CODEC_JSON, page=1, page_size=None):
        self.default_format = format
        self.default_page  = page
        self.default_page_size  = page_size if page_size else get_page_size()

    @autohelp(True)
    def mRNASeq(self, **kwargs):
        '''
        This service returns sample-level log2 mRNASeq expression
        values. Results may be filtered by gene, cohort, barcode,
        sample type or characterization protocol, but at least one
        gene must be supplied.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Samples
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
            gene  (str)  Comma separated list of gene name(s).
            cohort  (str)  Narrow search to one or more TCGA disease cohorts.
            barcode  (str)  Comma separated list of TCGA participant barcodes (e.g. TCGA-GF-A4EO).
            sample_type  (str)  Narrow search to one or more TCGA sample types.
            protocol  (str)  Narrow search to one or more sample characterization protocols.
            page  (int)  Which page (slice) of entire results set should be returned. 
            page_size  (int)  Number of records per page of results.  Max is 2000.
            sort_by  (str)  Which column in the results should be used for sorting paginated results?
        '''

        vars = locals()
        spec = dict2obj({'defaults': [None],
            'pathParams': [],
            'queryParams': [u'format',
                           u'gene',
                           u'cohort',
                           'barcode',
                           u'sample_type',
                           u'protocol',
                           u'page',
                           u'page_size',
                           u'sort_by'],
            'requiredParams': [],
            'semiReqParams': [u'gene'],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Samples/mRNASeq"
        call += queryParams
        return get(call, **clientParams)

    @autohelp(True)
    def miRSeq(self, **kwargs):
        '''
        This service returns sample-level log2 miRSeq expression
        values. Results may be filtered by miR, cohort, barcode,
        sample type or Firehose preprocessing tool, but at least one
        miR must be supplied.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Samples
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
            mir  (str)  Comma separated list of miR names (e.g. hsa-let-7b-5p,hsa-let-7a-1).
            cohort  (str)  Narrow search to one or more TCGA disease cohorts.
            barcode  (str)  Comma separated list of TCGA participant barcodes (e.g. TCGA-GF-A4EO).
            tool  (str)  Narrow search to include only data/results produced by the selected Firehose tool.
            sample_type  (str)  Narrow search to one or more TCGA sample types.
            page  (int)  Which page (slice) of entire results set should be returned. 
            page_size  (int)  Number of records per page of results.  Max is 2000.
            sort_by  (str)  Which column in the results should be used for sorting paginated results?
        '''

        vars = locals()
        spec = dict2obj({'defaults': [None],
            'pathParams': [],
            'queryParams': [u'format',
                           u'mir',
                           u'cohort',
                           'barcode',
                           u'tool',
                           u'sample_type',
                           u'page',
                           u'page_size',
                           u'sort_by'],
            'requiredParams': [],
            'semiReqParams': [u'mir'],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Samples/miRSeq"
        call += queryParams
        return get(call, **clientParams)

    @autohelp(True)
    def Clinical(self, **kwargs):
        '''
        This service returns patient clinical data from TCGA,
        verbatim. It differs from the Samples/Clinical_FH method by
        providing access to all TCGA CDEs in their original form, not
        merely the subset of CDEs normalized by Firehose for analyses.
        Results may be selected by disease cohort, patient barcode or
        CDE name, but at least one cohort, barcode, or CDE must be
        provided. When filtering by CDE note that only when a patient
        record contains one or more of the selected CDEs will it be
        returned. Visit the Metadata/ClinicalNames api function to see
        the entire list of TCGA CDEs that may be queried via this
        method. For more information on how clinical data are
        processed, see our <a href="https://confluence.broadinstitute.
        org/display/GDAC/Documentation#Documentation-
        ClinicalPipeline">pipeline documentation</a>.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Samples
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
            cohort  (str)  Narrow search to one or more TCGA disease cohorts.
            barcode  (str)  Comma separated list of TCGA participant barcodes (e.g. TCGA-GF-A4EO).
            cde  (str)  Retrieve results only for specified CDEs, per the Metadata/ClinicalNames function
            page  (int)  Which page (slice) of entire results set should be returned. 
            page_size  (int)  Number of records per page of results.  Max is 2000.
            sort_by  (str)  Which column in the results should be used for sorting paginated results?
        '''

        vars = locals()
        spec = dict2obj({'defaults': [None, None, None],
            'pathParams': [],
            'queryParams': [u'format',
                           u'cohort',
                           'barcode',
                           'cde',
                           u'page',
                           u'page_size',
                           u'sort_by'],
            'requiredParams': [],
            'semiReqParams': [u'cohort', 'barcode', 'cde'],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Samples/Clinical"
        call += queryParams
        return get(call, **clientParams)

    @autohelp(True)
    def Clinical_FH(self, **kwargs):
        '''
        This service returns patient-level clinical data elements
        (CDEs) that have been normalized by Firehose for use in
        analyses. Results may be selected by disease cohort, patient
        barcode or CDE name, but at least one cohort, barcode or CDE
        must be provided. When filtering by CDE note that only when a
        patient record contains one or more of the selected CDEs will
        it be returned. Visit <a
        href="http://gdac.broadinstitute.org/runs/info/clinical">this
        table of CDEs</a> to see what's available for every disase
        cohort; for more information on how these CDEs are processed
        see our <a href="https://confluence.broadinstitute.org/display
        /GDAC/Documentation#Documentation-ClinicalPipeline">pipeline
        documentation</a>.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Samples
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
            cohort  (str)  Narrow search to one or more TCGA disease cohorts.
            barcode  (str)  Comma separated list of TCGA participant barcodes (e.g. TCGA-GF-A4EO).
            fh_cde  (str)  Retrieve results only for the CDEs specified.
            page  (int)  Which page (slice) of entire results set should be returned. 
            page_size  (int)  Number of records per page of results.  Max is 2000.
            sort_by  (str)  Which column in the results should be used for sorting paginated results?
        '''

        vars = locals()
        spec = dict2obj({'defaults': [None, None, None],
            'pathParams': [],
            'queryParams': [u'format',
                           u'cohort',
                           'barcode',
                           'fh_cde',
                           u'page',
                           u'page_size',
                           u'sort_by'],
            'requiredParams': [],
            'semiReqParams': [u'cohort', 'barcode', 'fh_cde'],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Samples/Clinical_FH"
        call += queryParams
        return get(call, **clientParams)

class Analyses (object):
    ''' Fine grained retrieval of analysis pipeline results '''

    def __init__(self, format=CODEC_JSON, page=1, page_size=None):
        self.default_format = format
        self.default_page  = page
        self.default_page_size  = page_size if page_size else get_page_size()

    @autohelp(True)
    def MutationMAF(self, **kwargs):
        '''
        This service returns columns from the MAF generated by MutSig.
        Results may be filtered by gene, cohort, tool, or barcode, but
        at least one gene OR barcode OR cohort must be given.  By
        default a subset consisting of the most commonly used columns
        will be returned, but that can be modified with the column
        parameter. Specifying 'all' in this parameter is a convenient
        way to return every column of the respective MAF, and has
        precedence over any any other column selection expression.
        The complete list of column names that may be specified is <a
        href='http://firebrowse.org/api-
        docs/#!/Metadata/MAFColNames'>given here</a>.  For more
        information on the mutation data, and how it is processed by
        Firehose, please consult the <a href="https://confluence.broad
        institute.org/display/GDAC/Documentation#Documentation-
        MutationPipelines">pipeline documentation</a>.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Analyses
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
            cohort  (str)  Narrow search to one or more TCGA disease cohorts.
            tool  (str)  Narrow search to include only data/results produced by the selected Firehose tool.
            gene  (str)  Comma separated list of gene name(s).
            barcode  (str)  Comma separated list of TCGA participant barcodes (e.g. TCGA-GF-A4EO).
            column  (str)  Comma separated list of which data columns/fields to return.
            page  (int)  Which page (slice) of entire results set should be returned. 
            page_size  (int)  Number of records per page of results.  Max is 2000.
            sort_by  (str)  Which column in the results should be used for sorting paginated results?
        '''

        vars = locals()
        spec = dict2obj({'defaults': [None, None, None],
            'pathParams': [],
            'queryParams': [u'format',
                           u'cohort',
                           u'tool',
                           u'gene',
                           'barcode',
                           u'column',
                           u'page',
                           u'page_size',
                           u'sort_by'],
            'requiredParams': [],
            'semiReqParams': [u'cohort', u'gene', 'barcode'],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Analyses/Mutation/MAF"
        call += queryParams
        return get(call, **clientParams)

    @autohelp(True)
    def MutationSMG(self, **kwargs):
        '''
        This service provides a list of significantly mutated genes,
        as scored by MutSig.  It may be filtered by cohort, rank,
        gene, tool and/or Q-value threshold, but at least one cohort
        or gene must be supplied.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Analyses
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
            cohort  (str)  Narrow search to one or more TCGA disease cohorts.
            tool  (str)  Narrow search to include only data/results produced by the selected Firehose tool.
            rank  (int)  Number of significant genes to return.
            gene  (str)  Comma separated list of gene name(s).
            q  (float)  Only return results with Q-value <= given threshold.
            page  (int)  Which page (slice) of entire results set should be returned. 
            page_size  (int)  Number of records per page of results.  Max is 2000.
            sort_by  (str)  Which column in the results should be used for sorting paginated results?
        '''

        vars = locals()
        spec = dict2obj({'defaults': [None, None],
            'pathParams': [],
            'queryParams': [u'format',
                           u'cohort',
                           u'tool',
                           u'rank',
                           u'gene',
                           u'q',
                           u'page',
                           u'page_size',
                           u'sort_by'],
            'requiredParams': [],
            'semiReqParams': [u'cohort', u'gene'],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Analyses/Mutation/SMG"
        call += queryParams
        return get(call, **clientParams)

    @autohelp(True)
    def CopyNumberGenesAll(self, **kwargs):
        '''
        This service provides access to the Gistic2
        all_data_by_genes.txt output data. This data is a gene-level
        table of copy number values for all samples. The returned copy
        number values are in units (copy number - 2) so that no
        amplification or deletion is 0, genes with amplifications have
        positive values, and genes with deletions are negative values.
        The data are converted from marker level to gene level using
        the extreme method: a gene is assigned the greatest
        amplification or the least deletion value among the markers it
        covers. Results may be filtered by cohort, gene, or barcode,
        but at least one gene or barcode must be supplied.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Analyses
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
            cohort  (str)  Narrow search to one or more TCGA disease cohorts.
            gene  (str)  Comma separated list of gene name(s).
            barcode  (str)  Comma separated list of TCGA participant barcodes (e.g. TCGA-GF-A4EO).
            page  (int)  Which page (slice) of entire results set should be returned. 
            page_size  (int)  Number of records per page of results.  Max is 2000.
            sort_by  (str)  Which column in the results should be used for sorting paginated results?
        '''

        vars = locals()
        spec = dict2obj({'defaults': [None, None],
            'pathParams': [],
            'queryParams': [u'format',
                           u'cohort',
                           u'gene',
                           'barcode',
                           u'page',
                           u'page_size',
                           u'sort_by'],
            'requiredParams': [],
            'semiReqParams': [u'gene', 'barcode'],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Analyses/CopyNumber/Genes/All"
        call += queryParams
        return get(call, **clientParams)

    @autohelp(True)
    def CopyNumberGenesFocal(self, **kwargs):
        '''
        This service provides access to the Gistic2
        focal_data_by_genes.txt output data. This output is similar to
        the all_data_by_genes.txt output, but using only focal events
        with lengths greater than the  focal length cutoff. This data
        is a gene-level table of copy number values for all samples.
        The returned copy number values are in units (copy number - 2)
        so that no amplification or deletion is 0, genes with
        amplifications have positive values, and genes with deletions
        are negative values. The data are converted from marker level
        to gene level using the extreme method: a gene is assigned the
        greatest amplification or the least deletion value among the
        markers it covers. Results may be filtered by cohort, gene,
        and/or barcode, but at least one gene or barcode must be
        supplied.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Analyses
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
            cohort  (str)  Narrow search to one or more TCGA disease cohorts.
            gene  (str)  Comma separated list of gene name(s).
            barcode  (str)  Comma separated list of TCGA participant barcodes (e.g. TCGA-GF-A4EO).
            page  (int)  Which page (slice) of entire results set should be returned. 
            page_size  (int)  Number of records per page of results.  Max is 2000.
            sort_by  (str)  Which column in the results should be used for sorting paginated results?
        '''

        vars = locals()
        spec = dict2obj({'defaults': [None, None],
            'pathParams': [],
            'queryParams': [u'format',
                           u'cohort',
                           u'gene',
                           'barcode',
                           u'page',
                           u'page_size',
                           u'sort_by'],
            'requiredParams': [],
            'semiReqParams': [u'gene', 'barcode'],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Analyses/CopyNumber/Genes/Focal"
        call += queryParams
        return get(call, **clientParams)

    @autohelp(True)
    def CopyNumberGenesThresholded(self, **kwargs):
        '''
        This service provides access to the Gistic2
        all_thresholded_by_genes.txt output data. A gene-level table
        of discrete amplification and deletion indicators for all
        samples. A table value of 0 means no amplification or deletion
        above the threshold. Amplifications are positive numbers: 1
        means amplification above the amplification threshold; 2 means
        amplifications larger to the arm level amplifications observed
        for the sample. Deletions are represented by negative table
        values: -1 represents deletion beyond the threshold; -2 means
        deletions greater than the minimum arm-level deletion observed
        for the sample. Results maybe filtered by cohort, gene or
        barcode, but at least one gene or barcode must be supplied.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Analyses
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
            cohort  (str)  Narrow search to one or more TCGA disease cohorts.
            gene  (str)  Comma separated list of gene name(s).
            barcode  (str)  Comma separated list of TCGA participant barcodes (e.g. TCGA-GF-A4EO).
            page  (int)  Which page (slice) of entire results set should be returned. 
            page_size  (int)  Number of records per page of results.  Max is 2000.
            sort_by  (str)  Which column in the results should be used for sorting paginated results?
        '''

        vars = locals()
        spec = dict2obj({'defaults': [None, None],
            'pathParams': [],
            'queryParams': [u'format',
                           u'cohort',
                           u'gene',
                           'barcode',
                           u'page',
                           u'page_size',
                           u'sort_by'],
            'requiredParams': [],
            'semiReqParams': [u'gene', 'barcode'],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Analyses/CopyNumber/Genes/Thresholded"
        call += queryParams
        return get(call, **clientParams)

    @autohelp(True)
    def CopyNumberGenesAmplified(self, **kwargs):
        '''
        This service provides access to the Gistic2
        amp_genes.conf_99.txt output data.  At least 1 gene or cohort
        must be supplied.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Analyses
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
            cohort  (str)  Narrow search to one or more TCGA disease cohorts.
            gene  (str)  Comma separated list of gene name(s).
            q  (float)  Only return results with Q-value <= given threshold.
            page  (int)  Which page (slice) of entire results set should be returned. 
            page_size  (int)  Number of records per page of results.  Max is 2000.
            sort_by  (str)  Which column in the results should be used for sorting paginated results?
        '''

        vars = locals()
        spec = dict2obj({'defaults': [None, None],
            'pathParams': [],
            'queryParams': [u'format',
                           u'cohort',
                           u'gene',
                           u'q',
                           u'page',
                           u'page_size',
                           u'sort_by'],
            'requiredParams': [],
            'semiReqParams': [u'cohort', u'gene'],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Analyses/CopyNumber/Genes/Amplified"
        call += queryParams
        return get(call, **clientParams)

    @autohelp(True)
    def CopyNumberGenesDeleted(self, **kwargs):
        '''
        This service provides access to the Gistic2
        del_genes.conf_99.txt output data.  At least 1 gene or cohort
        must be supplied.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Analyses
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
            cohort  (str)  Narrow search to one or more TCGA disease cohorts.
            gene  (str)  Comma separated list of gene name(s).
            q  (float)  Only return results with Q-value <= given threshold.
            page  (int)  Which page (slice) of entire results set should be returned. 
            page_size  (int)  Number of records per page of results.  Max is 2000.
            sort_by  (str)  Which column in the results should be used for sorting paginated results?
        '''

        vars = locals()
        spec = dict2obj({'defaults': [None, None],
            'pathParams': [],
            'queryParams': [u'format',
                           u'cohort',
                           u'gene',
                           u'q',
                           u'page',
                           u'page_size',
                           u'sort_by'],
            'requiredParams': [],
            'semiReqParams': [u'cohort', u'gene'],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Analyses/CopyNumber/Genes/Deleted"
        call += queryParams
        return get(call, **clientParams)

    @autohelp(False)
    def mRNASeqQuartiles(self, gene, **kwargs):
        '''
        For a given gene compute quartiles and extrema, suitable e.g.
        for drawing a boxplot (Tukey 1977).  Results may be filtered
        by cohort, sample type, patient barcode  or characterization
        protocol, and are returned sorted by cohort.  Note that
        samples for which no expression value was recorded (e.g. <a hr
        ef="http://firebrowse.org/api/v1/Samples/mRNASeq?&gene=egfr&co
        hort=SKCM&tcga_participant_barcode=TCGA-GN-A262">the melanoma
        sample TCGA-GN-262</a>) are removed prior to calculation.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Analyses
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
            gene  (str)  Enter a single gene name.
            cohort  (str)  Narrow search to one or more TCGA disease cohorts.
            protocol  (str)  Narrow search to one or more sample characterization protocols.
            sample_type  (str)  For which type of sample(s) should quartiles be computed?
            Exclude  (str)  Comma separated list of TCGA participants, identified by barcodes such as TCGA-GF-A4EO, denoting samples to exclude from computation.
        '''

        vars = locals()
        spec = dict2obj({'defaults': [],
            'pathParams': [],
            'queryParams': [u'format',
                           u'gene',
                           u'cohort',
                           u'protocol',
                           u'sample_type',
                           u'Exclude'],
            'requiredParams': [u'gene'],
            'semiReqParams': [],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Analyses/mRNASeq/Quartiles"
        call += queryParams
        return get(call, **clientParams)

    @autohelp(False)
    def Reports(self, **kwargs):
        '''
        This service returns URLs to the analysis result reports for
        runs of the Broad Institute GDAC Firehose analysis pipeline.
        At least one year of run reports are maintained in the
        database, but the reports from the latest run will be returned
        by default. The set of <a href="https://confluence.broadinstit
        ute.org/display/GDAC/Nozzle">Nozzle</a> reports returned may
        be filtered by disease cohort, report type and report name.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Analyses
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
            date  (str)  Select one or more date stamps.
            cohort  (str)  Narrow search to one or more TCGA disease cohorts.
            name  (str)  Narrow search to one or more report names.
            type  (str)  Narrow search to one or more report types.
            page  (int)  Which page (slice) of entire results set should be returned. 
            page_size  (int)  Number of records per page of results.  Max is 2000.
            sort_by  (str)  Which column in the results should be used for sorting paginated results?
        '''

        vars = locals()
        spec = dict2obj({'defaults': [],
            'pathParams': [],
            'queryParams': [u'format',
                           u'date',
                           u'cohort',
                           u'name',
                           u'type',
                           u'page',
                           u'page_size',
                           u'sort_by'],
            'requiredParams': [],
            'semiReqParams': [],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Analyses/Reports"
        call += queryParams
        return get(call, **clientParams)

    @autohelp(True)
    def FeatureTable(self, **kwargs):
        '''
        This service returns part or all of the so-called
        <strong>feature table</strong>; which aggregates the most
        important findings across ALL pipelines in the GDAC Firehose
        analysis workflow into a single table for simple access.  One
        feature table is created per disease cohort.  Results may be
        filtered by date or cohort, but at least 1 cohort must be
        specified here. For more details please visit the <a href=http
        s://confluence.broadinstitute.org/display/GDAC/Documentation
        \#Documentation-FeatureTable>online documentation</a>.  Please
        note that the service is still undergoing experimental
        evaluation and does not return JSON format.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Analyses
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
            cohort  (str)  Narrow search to one or more TCGA disease cohorts.
            date  (str)  Select one or more date stamps.
            column  (str)  Comma separated list of which data columns/fields to return.
            page  (int)  Which page (slice) of entire results set should be returned. 
            page_size  (int)  Number of records per page of results.  Max is 2000.
        '''

        vars = locals()
        spec = dict2obj({'defaults': [None],
            'pathParams': [],
            'queryParams': [u'format',
                           u'cohort',
                           u'date',
                           u'column',
                           u'page',
                           u'page_size'],
            'requiredParams': [],
            'semiReqParams': [u'cohort'],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Analyses/FeatureTable"
        call += queryParams
        return get(call, **clientParams)

class Archives (object):
    ''' Bulk retrieval of data or analysis pipeline results '''

    def __init__(self, format=CODEC_JSON, page=1, page_size=None):
        self.default_format = format
        self.default_page  = page
        self.default_page_size  = page_size if page_size else get_page_size()

    @autohelp(False)
    def StandardData(self, **kwargs):
        '''
        This service returns the archive URLs for our Firehose
        standard data runs, providing a RESTful interface similar in
        spirit to the command line <a href="https://confluence.broadin
        stitute.org/display/GDAC/Download">firehose_get</a> tool. The
        archives can be filtered based on date, cohort, data type,
        platform, center, data level, and protocol.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Archives
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
            date  (str)  Select one or more date stamps.
            cohort  (str)  Narrow search to one or more TCGA disease cohorts.
            data_type  (str)  Narrow search to one or more TCGA data types.
            tool  (str)  Narrow search to include only data/results produced by the selected Firehose tool.
            platform  (str)  Narrow search to one or more TCGA data generation platforms.
            center  (str)  Narrow search to one or more TCGA centers.
            level  (int)  Narrow search to one or more TCGA data levels.
            protocol  (str)  Narrow search to one or more sample characterization protocols.
            page  (int)  Which page (slice) of entire results set should be returned. 
            page_size  (int)  Number of records per page of results.  Max is 2000.
            sort_by  (str)  Which column in the results should be used for sorting paginated results?
        '''

        vars = locals()
        spec = dict2obj({'defaults': [],
            'pathParams': [],
            'queryParams': [u'format',
                           u'date',
                           u'cohort',
                           u'data_type',
                           u'tool',
                           u'platform',
                           u'center',
                           u'level',
                           u'protocol',
                           u'page',
                           u'page_size',
                           u'sort_by'],
            'requiredParams': [],
            'semiReqParams': [],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Archives/StandardData"
        call += queryParams
        return get(call, **clientParams)

class Metadata (object):
    ''' Retrieve disease, sample, and datatype descriptions, sample counts, and more '''

    def __init__(self, format=CODEC_JSON, page=1, page_size=None):
        self.default_format = format
        self.default_page  = page
        self.default_page_size  = page_size if page_size else get_page_size()

    @autohelp(False)
    def Centers(self, **kwargs):
        '''
        By default this function returns a table of all consortium
        members in TCGA, aka centers; it provides both the HTTP domain
        and full organizational name of each center.  A subset of this
        table may be obtained by explicitly specifying one or more
        domain names.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Metadata
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
            center  (str)  Narrow search to one or more TCGA centers.
        '''

        vars = locals()
        spec = dict2obj({'defaults': [],
            'pathParams': [],
            'queryParams': [u'format', u'center'],
            'requiredParams': [],
            'semiReqParams': [],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Metadata/Centers"
        call += queryParams
        return get(call, **clientParams)

    @autohelp(False)
    def ClinicalNames(self, **kwargs):
        '''
        Retrieve names of all patient-level clinical data elements
        (CDES) available in TCGA, unioned across all disease cohorts.
        A CDE will be listed here only when it has a value other than
        NA for at least 1 patient case in any disease cohort. For more
        information on how these CDEs are processed see our <a href="h
        ttps://confluence.broadinstitute.org/display/GDAC/Documentatio
        n">pipeline documentation</a>.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Metadata
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
        '''

        vars = locals()
        spec = dict2obj({'defaults': [],
            'pathParams': [],
            'queryParams': [u'format'],
            'requiredParams': [],
            'semiReqParams': [],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Metadata/ClinicalNames"
        call += queryParams
        return get(call, **clientParams)

    @autohelp(False)
    def ClinicalNames_FH(self, **kwargs):
        '''
        This service returns the names of patient-level clinical data
        elements (CDEs) that have been normalized by Firehose for use
        in analyses, unioned across all disease cohorts. For more
        information on how these CDEs are processed, see our <a href="
        https://confluence.broadinstitute.org/display/GDAC/Documentati
        on">pipeline documentation</a>.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Metadata
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
        '''

        vars = locals()
        spec = dict2obj({'defaults': [],
            'pathParams': [],
            'queryParams': [u'format'],
            'requiredParams': [],
            'semiReqParams': [],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Metadata/ClinicalNames_FH"
        call += queryParams
        return get(call, **clientParams)

    @autohelp(False)
    def Cohorts(self, **kwargs):
        '''
        By default this function returns a table containing all TCGA
        cohort abbreviations and their corresponding disease names.  A
        subset of that table may be obtained by explicitly specifying
        one or more cohort abbreviations.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Metadata
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
            cohort  (str)  Narrow search to one or more TCGA disease cohorts.
        '''

        vars = locals()
        spec = dict2obj({'defaults': [],
            'pathParams': [],
            'queryParams': [u'format', u'cohort'],
            'requiredParams': [],
            'semiReqParams': [],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Metadata/Cohorts"
        call += queryParams
        return get(call, **clientParams)

    @autohelp(False)
    def Counts(self, **kwargs):
        '''
        Returns the aliquot counts for each disease cohort, per sample
        type and data type.  The sample type designation of "Tumor"
        may be used to aggregate the count of all tumor aliquots into
        a single number per disease and data type. See the SampleTypes
        function for a complete description of sample types.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Metadata
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
            date  (str)  Select one or more date stamps.
            cohort  (str)  Narrow search to one or more TCGA disease cohorts.
            sample_type  (str)  Narrow search to one or more TCGA sample types.
            data_type  (str)  Narrow search to one or more TCGA data types.
            totals  (bool)  Output an entry providing the totals for each data type.
            sort_by  (str)  Which column in the results should be used for sorting paginated results?
        '''

        vars = locals()
        spec = dict2obj({'defaults': [],
            'pathParams': [],
            'queryParams': [u'format',
                           u'date',
                           u'cohort',
                           u'sample_type',
                           u'data_type',
                           u'totals',
                           u'sort_by'],
            'requiredParams': [],
            'semiReqParams': [],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Metadata/Counts"
        call += queryParams
        return get(call, **clientParams)

    @autohelp(False)
    def Dates(self, **kwargs):
        '''
        Retrieve datestamps of all GDAC Firehose standard data and
        analyses runs that have been ingested into FireBrowse.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Metadata
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
        '''

        vars = locals()
        spec = dict2obj({'defaults': [],
            'pathParams': [],
            'queryParams': [u'format'],
            'requiredParams': [],
            'semiReqParams': [],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Metadata/Dates"
        call += queryParams
        return get(call, **clientParams)

    @autohelp(False)
    def HeartBeat(self, **kwargs):
        '''
        Returns a message to indicate that API (server) is up and
        running, or times out if not.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Metadata
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
        '''

        vars = locals()
        spec = dict2obj({'defaults': [],
            'pathParams': [],
            'queryParams': [u'format'],
            'requiredParams': [],
            'semiReqParams': [],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Metadata/HeartBeat"
        call += queryParams
        return get(call, **clientParams)

    @autohelp(False)
    def MAFColNames(self, **kwargs):
        '''
        Retrieve the names of all columns in the mutation annotation
        files (MAFs) hosted by FireBrowse.  For more information on
        the mutation data, and how it is processed by Firehose, please
        consult the <a href="https://confluence.broadinstitute.org/dis
        play/GDAC/Documentation#Documentation-
        MutationPipelines">pipeline documentation</a>.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Metadata
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
        '''

        vars = locals()
        spec = dict2obj({'defaults': [],
            'pathParams': [],
            'queryParams': [u'format'],
            'requiredParams': [],
            'semiReqParams': [],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Metadata/MAFColNames"
        call += queryParams
        return get(call, **clientParams)

    @autohelp(False)
    def Patients(self, **kwargs):
        '''
        This service returns a list of all TCGA patient barcodes in
        FireBrowse, optionally filtered by disease cohort.  The
        barcodes are obtained directy from the clinical data.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Metadata
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
            cohort  (str)  Narrow search to one or more TCGA disease cohorts.
            page  (int)  Which page (slice) of entire results set should be returned. 
            page_size  (int)  Number of records per page of results.  Max is 2000.
            sort_by  (str)  Which column in the results should be used for sorting paginated results?
        '''

        vars = locals()
        spec = dict2obj({'defaults': [],
            'pathParams': [],
            'queryParams': [u'format', u'cohort', u'page', u'page_size', u'sort_by'],
            'requiredParams': [],
            'semiReqParams': [],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Metadata/Patients"
        call += queryParams
        return get(call, **clientParams)

    @autohelp(False)
    def Platforms(self, **kwargs):
        '''
        By default this function returns a table of all of the
        technology platforms used to sequence or characterize samples
        in TCGA--both their short platform codes and full names.  A
        subset of this table may be obtained by explicitly specifying
        one or more platform codes.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Metadata
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
            platform  (str)  Narrow search to one or more TCGA data generation platforms.
        '''

        vars = locals()
        spec = dict2obj({'defaults': [],
            'pathParams': [],
            'queryParams': [u'format', u'platform'],
            'requiredParams': [],
            'semiReqParams': [],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Metadata/Platforms"
        call += queryParams
        return get(call, **clientParams)

    @autohelp(False)
    def SampleTypes(self, **kwargs):
        '''
        Return all TCGA sample type codes, both numeric and symbolic.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Metadata
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
        '''

        vars = locals()
        spec = dict2obj({'defaults': [],
            'pathParams': [],
            'queryParams': [u'format'],
            'requiredParams': [],
            'semiReqParams': [],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Metadata/SampleTypes"
        call += queryParams
        return get(call, **clientParams)

    @autohelp(False)
    def SampleTypeBarcode(self, TCGA_Barcode, **kwargs):
        '''
        Given a TCGA barcode, return its short letter sample type
        code.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Metadata
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            TCGA_Barcode  (str)  Enter a single TCGA barcode, of any form: e.g. TCGA-GF-A4EO-06 or TCGA-EL-A3D5-01A-22D-A202-08
            format  (str)  Format of result.
        '''

        vars = locals()
        spec = dict2obj({'defaults': [],
            'pathParams': [u'TCGA_Barcode'],
            'queryParams': [u'format'],
            'requiredParams': [u'TCGA_Barcode'],
            'semiReqParams': [],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Metadata/SampleType/Barcode"
        call += "/%s" % "/".join([vars[p] for p in spec.pathParams])
        call += queryParams
        return get(call, **clientParams)

    @autohelp(False)
    def SampleTypeCode(self, code, **kwargs):
        '''
        Convert a TCGA numeric sample type code (e.g. 01, 02) to its
        corresponding symbolic (short letter) code (e.g. TP, TR).

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Metadata
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            code  (str)  Narrow search to one or more TCGA sample type codes.
            format  (str)  Format of result.
        '''

        vars = locals()
        spec = dict2obj({'defaults': [],
            'pathParams': [u'code'],
            'queryParams': [u'format'],
            'requiredParams': [u'code'],
            'semiReqParams': [],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Metadata/SampleType/Code"
        call += "/%s" % "/".join([vars[p] for p in spec.pathParams])
        call += queryParams
        return get(call, **clientParams)

    @autohelp(False)
    def SampleTypeShortLetterCode(self, short_letter_code, **kwargs):
        '''
        Convert a TCGA sample type code in symbolic form (or 'short
        letter code' like TP, TR) to its corresponding numeric form
        (e.g. 01, 02).

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Metadata
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            short_letter_code  (str)  TCGA sample type short letter code(s) (e.g. TP, NB, etc.). 
            format  (str)  Format of result.
        '''

        vars = locals()
        spec = dict2obj({'defaults': [],
            'pathParams': [u'short_letter_code'],
            'queryParams': [u'format'],
            'requiredParams': [u'short_letter_code'],
            'semiReqParams': [],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Metadata/SampleType/ShortLetterCode"
        call += "/%s" % "/".join([vars[p] for p in spec.pathParams])
        call += queryParams
        return get(call, **clientParams)

    @autohelp(False)
    def TSSites(self, **kwargs):
        '''
        By default this function returns a table of all sites which
        contributed source tissue to TCGA, aka TSS's. A subset of this
        table may be obtained by explicitly specifying one or more
        sites.

        For more details consult the interactive documentation at
            http://firebrowse.org/api-docs/#!/Metadata
        OR use help(param_values) to see the range of values accepted
        for each parameter, the defaults for each (if any), and the
        degrees of optionality/requiredness offered by the API.

        Parameters: 
            format  (str)  Format of result.
            tss_code  (str)  Narrow search to one or more TSS codes
        '''

        vars = locals()
        spec = dict2obj({'defaults': [],
            'pathParams': [],
            'queryParams': [u'format', u'tss_code'],
            'requiredParams': [],
            'semiReqParams': [],
            'caller' : self})
        queryParams, clientParams = parse_args(vars, spec)
        call = "api/v1/Metadata/TSSites"
        call += queryParams
        return get(call, **clientParams)

